<?php
header('Content-Type: application/json'); // Set header for JSON response
include '../../config.php';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        $data = json_decode(file_get_contents("php://input"), true);

        if (!isset($data['user_id']) || !isset($data['at_time'])) {
            throw new Exception("All mandatory fields are required");
        }

        $user_id = $data['user_id'];
        $location = $data['location'] ?? null;
        $at_time = $data['at_time']; // Ensure this is in ISO 8601 UTC format (e.g., 2024-11-19T08:25:15.524Z)

        // Configurable variables
        $timeDifferenceAllowed = 900; // Allowed time difference in seconds (e.g., 10 minutes)
        $deleteAfterDays = 60; // Days after which records should be deleted

        $conn->beginTransaction(); // Start transaction

        // Delete records older than the configured number of days
        $deleteSql = "DELETE FROM user_location 
                      WHERE user_id = :user_id 
                      AND CONVERT_TZ(STR_TO_DATE(at_time, '%Y-%m-%dT%H:%i:%s.%fZ'), '+00:00', '+00:00') < UTC_TIMESTAMP() - INTERVAL :deleteAfterDays DAY";

        $deleteStmt = $conn->prepare($deleteSql);
        $deleteStmt->execute([
            ':user_id' => $user_id,
            ':deleteAfterDays' => $deleteAfterDays
        ]);

        // Check if a record exists within the allowed time difference
        $checkSql = "SELECT at_time FROM user_location 
                     WHERE user_id = :user_id 
                     ORDER BY CONVERT_TZ(STR_TO_DATE(at_time, '%Y-%m-%dT%H:%i:%s.%fZ'), '+00:00', '+00:00') DESC 
                     LIMIT 1";

        $checkStmt = $conn->prepare($checkSql);
        $checkStmt->execute([':user_id' => $user_id]);
        $lastRecord = $checkStmt->fetch(PDO::FETCH_ASSOC);

        if ($lastRecord) {
            $lastAtTime = strtotime($lastRecord['at_time']); // Convert last at_time to timestamp
            $currentAtTime = strtotime($at_time); // Convert input at_time to timestamp
            $timeDifference = $currentAtTime - $lastAtTime;

            if ($timeDifference < $timeDifferenceAllowed) {
                throw new Exception("A record already exists within the allowed time difference.");
            }
        }

        // Insert the new record into the user_location table
        $sql = "INSERT INTO user_location (user_id, location, at_time)
                VALUES (:user_id, :location, :at_time)";

        $stmt = $conn->prepare($sql);
        $stmt->execute([
            ':user_id' => $user_id,
            ':location' => $location,
            ':at_time' => $at_time // Stored as TEXT in UTC
        ]);

        $conn->commit(); // Commit transaction

        $response = [
            "success" => true,
            "message" => "New record created successfully"
        ];
    } catch (PDOException $e) {
        $conn->rollBack(); // Rollback transaction
        $response = [
            "success" => false,
            "message" => "Database error occurred",
            "error_details" => $e->getMessage()
        ];
    } catch (Exception $e) {
        $response = [
            "success" => false,
            "message" => $e->getMessage(),
            "error_details" => $e->getMessage()
        ];
    }
    echo json_encode($response);
} else {
    echo json_encode([
        "success" => false,
        "message" => "Invalid request method"
    ]);
}
